/*****************************************************************************/
/*  Copyright 1993 National Instruments Corporatation. All rights reserved.  */
/*****************************************************************************/

#include <utility.h>
#include <gpib.h>
#include <formatio.h>
#include <nivxi.h>
#include "tkvx433X.h"

/*  =========================================================================  */
/*  = UTILITY ROUTINES ======================================================  */
int tkvx433X_open_instr (int, int, int *);
int tkvx433X_close_instr (int);
int tkvx433X_init_gpib (int);
int tkvx433X_invalid_integer_range (int, int, int, int);
int tkvx433X_invalid_longint_range (long, long, long, int);
int tkvx433X_device_closed (int);
int tkvx433X_read_data (int, char *, int);
int tkvx433X_write_data (int, char *, int);
int tkvx433X_read_reg (int, int, int *);
void tkvx433X_setup_arrays (void);
int tkvx433X_count_modules (int, int *);
int tkvx433X_find_relay_range (int, int, int *);

/*  = Tektronix VX433X/VX437X Master/Slave Module ===========================  */
/*   LabWindows 2.2 Instrument Driver                                          */
/*   Original Release: July 1992                                               */
/*   By: HP, National Instruments                                              */
/*   Originally written in C                                                   */
/*   Modification History:                                                     */
/*            04/20/94 - Changed the GPIB Library calls and Word Serial        */
/*                       VXI Library calls so that they check for the error    */
/*                       bit being set to indicate an error (& 0x8000)         */
/*                       instead of checking for a negative return value.      */
/*                       Changed the VXI "read_reg" function so that           */
/*                       the type conflict which existed because of the        */
/*                       unsigned pointer is resolved, fixed by changing       */
/*                       to an unsigned short and passing its address.         */
/*                       Modified by GW, NI Austin.                            */
/*                                                                             */                                                                          
/*            07/08/96 - Modified for MDLL Compatibility                       */
/*                       DS, National Instruments, Austin, Texas               */
/*                       Phone (800) 433-3488  Fax (512) 794-5678              */
/*                                                                             */                                                                          
/*******************************************************************************/ 

/*  = INSTRUMENT TABLE ======================================================  */
/*   log_addr array: contains the logical addresses of opened instruments.     */
/*   bd array: contains the device descriptors returned by OpenDev (NI-GPIB only)   */
/*   interface array: specifies the type of interface (NI-VXI or NI-GPIB).     */
/*   base_addr array: contains the base address in A16 space for the instr.    */
/*   relays_per_mod table: contains the total number of modules connected to   */
/*                         each master scanner as well as the total number of  */
/*                         relays in each module.                              */
/*   instr_cnt: contains the number of instruments open of this model type.    */
/*   tkvx433X_err: the error variable for the instrument module.               */
/*  =========================================================================  */
static int  log_addr[tkvx433X_MAX_INSTR + 1];
static int  bd[tkvx433X_MAX_INSTR + 1];
static int  interface[tkvx433X_MAX_INSTR + 1];
static long  base_addr[tkvx433X_MAX_INSTR + 1];
static int  relays_per_mod[tkvx433X_MAX_INSTR + 1][13];
static int instr_cnt;
static int tkvx433X_err;
/*  = STATIC DATA ===========================================================  */
/*   cmd : a buffer for GPIB and VXI I/O strings.                              */
/*   timeout: a look up table used to associate GPIB timeout codes with Word   */
/*            Serial timout codes.                                             */
/*  =========================================================================  */
static char  cmd[100];
static long  timeout[18];
/*  = INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================  */
static char * switching[3];
static char * delaytime[4];
static char * interrupts[6];
static char * trig_type[7];

/*  =========================================================================  */
/*   Function: Initialize                                                      */
/*   Purpose:  This function opens the instrument, queries for ID, and         */
/*             initializes the instrument to a known state.                    */
/*  =========================================================================  */
int tkvx433X_init (int laddr, int platform, int *instrID, int scan_type)
{
    int id;
    int value;
    int max_relays;
    int module_cnt;

    if (tkvx433X_invalid_integer_range (laddr, 0, 255,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (platform, 1, 2,  -2) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (scan_type, 0, 1,  -4) != 0)
        return tkvx433X_err;
    /*   Initialize entry in Instrument Table and interface for instrument.   */
    if (tkvx433X_open_instr (laddr, platform, &id) != 0)
        return tkvx433X_err;
    /*   Read the instrument ID Register   */
    if (tkvx433X_read_reg (id, 0, &value) != 0)  {
        tkvx433X_close (id);
        tkvx433X_err = 231;
        return tkvx433X_err;
    }
    /*   Limit the value to bits 0-11   */
    value &= 0xfff;
    /*   Check the ID Register of the appropriate Master Scanner            */
    if (value != 0xffc)  {
        tkvx433X_close (id);
        tkvx433X_err = 223;
        return tkvx433X_err;
    }
    /*   Read the Device Type Register   */
    if (tkvx433X_read_reg (id, 2, &value) != 0)  {
        tkvx433X_close (id);
        tkvx433X_err = 231;
        return tkvx433X_err;
    }
    /*   Limit the value to bits 0-11   */
    value &= 0xfff;
    /*   Check the Device Type Register of the appropriate Master Scanner   */
    if (!scan_type && value != 0x6b3 || scan_type == 1 && value != 0x6b1)  {
        tkvx433X_close (id);
        tkvx433X_err = 223;
        return tkvx433X_err;
    }
    /*   Initialize the instrument to a known state.   */
    if (tkvx433X_write_data (id, "reset;", 6) != 0)  {
        tkvx433X_close_instr (id);
        return tkvx433X_err;
    }
    /*   Initialize Instrument-Dependent Command arrays.   */
    tkvx433X_setup_arrays ();
    *instrID = id;
    /*   Count the number of Slave Modules connected for a given master.   */
    if (tkvx433X_count_modules (id, &module_cnt) != 0)
        return tkvx433X_err;
    /*   store the number of modules   */
    relays_per_mod[id][12] = module_cnt;
    while (module_cnt !=  -1)  {
        /*   count number of relays per mod and store in array   */
        if (tkvx433X_find_relay_range (id, module_cnt, &max_relays) != 0)
            return tkvx433X_err;
        relays_per_mod[id][module_cnt] = max_relays;
        module_cnt--;
    }
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Reset                                                           */
/*   Purpose:  This function resets the instrument to its power-up conditions  */
/*  =========================================================================  */
int tkvx433X_reset (int instrID)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    if (tkvx433X_write_data (instrID, "reset;", 6) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Wait                                                            */
/*   Purpose:  This function makes the instrument wait a certain amount of     */
/*             milliseconds before accepting another command.                  */
/*  =========================================================================  */
int tkvx433X_wait (int instrID, int wait_time)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (wait_time, 0, 32767,  -2) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    Fmt (cmd, "%s<wait %d[b4];", wait_time);
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: List Error                                                      */
/*   Purpose:  This function reads the error query from the instrument and     */
/*             returns the error message.                                      */
/*  =========================================================================  */
int tkvx433X_list_error (int instrID, char err_msg[])
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    Fmt (cmd, "%s<list? errors;");
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    if (tkvx433X_read_data (instrID, cmd, 100) != 0)
        return tkvx433X_err;
    Fmt (err_msg, "%s<                                                 ");
    Fmt (err_msg, "%s<%s[t13]", cmd);
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Set Timeout                                                     */
/*   Purpose:  This function changes or disables the timeout of the device.    */
/*             Refer to array at the end of this file for timeout codes        */
/*             (index of timeout array).                                       */
/*  =========================================================================  */
int tkvx433X_set_timeout (int instrID, int tmo_code)
{
    long actual_tmo;

    tkvx433X_err = 0;
    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (tmo_code, 0, 17,  -2) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    switch (interface[instrID])  {
    case NI_VXI:
        if (WSsetTmo (timeout[tmo_code], &actual_tmo) < 0)
            tkvx433X_err = 239;
        break;
    case NI_GPIB:
        if (ibtmo (bd[instrID], tmo_code) & 0x8000)
            tkvx433X_err = 239;
        break;
    default:
        tkvx433X_err = 239;
        break;
    }
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Join Module                                                     */
/*   Purpose:  This function joins (or disjoins) the sections A and B of a     */
/*             given module.                                                   */
/*  =========================================================================  */
int tkvx433X_joinmodule (int instrID, int module, int fun)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (module, 0, relays_per_mod[instrID][12],  -2) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (fun, 0, 1,  -3) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    switch (fun)  {
    case 0:
        Fmt (cmd, "%s<disjoin %d[b4];", module);
        break;
    case 1:
        Fmt (cmd, "%s<join %d[b4];", module);
        break;
    }
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Switch Relays                                                   */
/*   Purpose:  This function closes (or opens) relays in a given module of an  */
/*             an instrument.                                                  */
/*  =========================================================================  */
int tkvx433X_switchrelays (int instrID, int module, int fun, int relaynum)
{
    int mod_rel_num;

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (fun, 0, 2,  -3) != 0)
        return tkvx433X_err;
    if (fun != 1)  {
        if (tkvx433X_invalid_integer_range (module, 0, relays_per_mod[instrID][12],  -2) != 0)
            return tkvx433X_err;
        if (tkvx433X_invalid_integer_range (relaynum, 0, relays_per_mod[instrID][module],  -4) != 0)
            return tkvx433X_err;
    }
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    switch (fun)  {
    case 0:
    case 2:
        mod_rel_num = module * 100 + relaynum;
        Fmt (cmd, "%s<%s %d[b4];", switching[fun], mod_rel_num);
        break;
    case 1:
        Fmt (cmd, "%s<%s", switching[fun]);
        break;
    }
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Change Delay Times                                              */
/*   Purpose:  This function changes the amount of time a given module will    */
/*             delay accepting another command after opening or closing a      */
/*             relay.                                                          */
/*  =========================================================================  */
int tkvx433X_changedelaytimes (int instrID, int fun, int module, long time)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (fun, 0, 3,  -3) != 0)
        return tkvx433X_err;
    /*   Check module range   */
    if (!fun || fun == 2)
        if (tkvx433X_invalid_integer_range (module, 0, relays_per_mod[instrID][12],  -2) != 0)
            return tkvx433X_err;
    if (tkvx433X_invalid_longint_range (time, 0L, 32767L,  -4) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    switch (fun)  {
    case 0:
    case 2:
        Fmt (cmd, "%s<%s %d[b4] %d[b4];", delaytime[fun], module, time);
        break;
    case 1:
    case 3:
        Fmt (cmd, "%s<%s %d[b4];", delaytime[fun], time);
        break;
    }
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Set Interrupts                                                  */
/*   Purpose:  This function sends commands to the instrument to enable or     */
/*             disable generating a Request True interrupt on the VXIbus on    */
/*             the occurrence of various events.                               */
/*  =========================================================================  */
int tkvx433X_set_interrupts (int instrID, int inter_type, int fun)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (inter_type, 0, 5,  -2) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (fun, 0, 1,  -3) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    switch (inter_type)  {
    case 0:
    case 1:
        /*   User interrupts   */
        Fmt (cmd, "%s<%s;", interrupts[inter_type]);
        break;
    default:
        switch (fun)  {
        case 0:
            Fmt (cmd, "%s<%s disable;", interrupts[inter_type]);
            break;
        case 1:
            Fmt (cmd, "%s<%s enable;", interrupts[inter_type]);
            break;
        }
        break;
    }
    /*   end inner switch   */
    /*   end outer switch   */
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Set Triggers                                                    */
/*   Purpose:  This function sends a command to the instrument to disable or   */
/*             define a specific VXIbus TTL trigger line for trigger in and    */
/*             trigger out commands and conditions.                            */
/*  =========================================================================  */
int tkvx433X_set_triggers (int instrID, int type, int line_num)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (type, 0, 6,  -2) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (line_num, 0, 7,  -3) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    switch (type)  {
    case 0:
    case 2:
    case 4:
        Fmt (cmd, "%s<%s %d[b4];", trig_type[type], line_num);
        break;
    case 1:
    case 3:
    case 5:
    case 6:
        Fmt (cmd, "%s<%s", trig_type[type]);
        break;
    }
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Input Program                                                   */
/*   Purpose:  This function sends a command to the instrument to put it in    */
/*             in Program Mode.                                                */
/*  =========================================================================  */
int tkvx433X_input_program (int instrID)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    Fmt (cmd, "%s<program input;");
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: End Program                                                     */
/*   Purpose:  This function sends a command to the instrument to end the      */
/*             the Program Mode.                                                */
/*  =========================================================================  */
int tkvx433X_end_program (int instrID)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    Fmt (cmd, "%s<program end;");
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Run Program                                                     */
/*   Purpose:  This function sends a command to the instrument to run the      */
/*             the program stored in its memory with the Program Input         */
/*             command.                                                        */
/*  =========================================================================  */
int tkvx433X_run_program (int instrID, int repetitions)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (repetitions, 0, 32767,  -2) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    Fmt (cmd, "%s<program go %d[b4];", repetitions);
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Pause Program                                                   */
/*   Purpose:  This function sends a command to the instrument to pause        */
/*             an executing program and return to accepting commands from      */
/*             the system controller.                                          */
/*  =========================================================================  */
int tkvx433X_pause_program (int instrID)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    Fmt (cmd, "%s<program pause;");
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Step Program                                                    */
/*   Purpose:  This function sends a command to the instrument to execute      */
/*             the program stored in its memory a certain number of steps at   */
/*             a time or to continue a paused program.                         */
/*  =========================================================================  */
int tkvx433X_step_program (int instrID, int num_steps)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_invalid_integer_range (num_steps, 0, 32767,  -2) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    Fmt (cmd, "%s<program step %d[b4];", num_steps);
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Close                                                           */
/*   Purpose:  This function closes the instrument.                            */
/*  =========================================================================  */
int tkvx433X_close (int instrID)
{

    if (tkvx433X_invalid_integer_range (instrID, 1, tkvx433X_MAX_INSTR,  -1) != 0)
        return tkvx433X_err;
    if (tkvx433X_device_closed (instrID) != 0)
        return tkvx433X_err;
    tkvx433X_close_instr (instrID);
    return tkvx433X_err;
}

/*  ====== UTILITY ROUTINES =================================================  */
/*  =========================================================================  */
/*   Function: Open Instrument                                                 */
/*   Purpose:  This function locates and initializes an entry in the           */
/*             Instrument Table.  The variable platform equals 1 for NI-VXI    */
/*             and 2 for NI-GPIB. The size of the Instrument Table can be      */
/*             changed in the include file by altering the constant            */
/*             tkvx433X_MAX_INSTR. The return value of this function is equal    */
/*             to the global error variable.                                   */
/*  =========================================================================  */
int tkvx433X_open_instr (int laddr, int platform, int *id)
{
    int i;
    int instrID;

    instrID = 0;
    tkvx433X_err = 0;
    /*   Check to see if the instrument is already in the Instrument Table.   */
    for (i = 1; i <= tkvx433X_MAX_INSTR; i++)
        if (log_addr[i] == laddr)  {
            instrID = i;
            i = tkvx433X_MAX_INSTR;
        }
    /*   If it is not in the instrument table, open an entry for the instrument.   */
    if (instrID <= 0)
        for (i = 1; i <= tkvx433X_MAX_INSTR; i++)
            if (!log_addr[i])  {
                instrID = i;
                log_addr[instrID] = laddr;
                base_addr[instrID] = (long)(laddr * 0x40) + 0xc000L;
                interface[instrID] = platform;
                instr_cnt++;
                i = tkvx433X_MAX_INSTR;
            }
    /*   If an entry could not be opened in the Instrument Table, return an error.  */
    if (instrID <= 0)  {
        tkvx433X_err = 220;
        return tkvx433X_err;
    }
    /*   Initialize the interface.   */
    switch (platform)  {
    case NI_VXI:
        if (InitVXIlibrary () < 0)  {
            tkvx433X_close_instr (instrID);
            tkvx433X_err = 222;
            return tkvx433X_err;
        }
        break;
    case NI_GPIB:
        if (tkvx433X_init_gpib (instrID) != 0)  {
            tkvx433X_close_instr (instrID);
            return tkvx433X_err;
        }
        break;
    }
    *id = instrID;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Close Instrument                                                */
/*   Purpose:  This function closes the instrument by removing it from the     */
/*             GPIB device table, if applicable (NI-GPIB only), and setting    */
/*             the logical address, bd, and interface to zero in the           */
/*             Instrument Table. Upon successful completion, zero is           */
/*             returned, otherwise -1 is returned.                             */
/*  =========================================================================  */
int tkvx433X_close_instr (int instrID)
{

    if (interface[instrID] == NI_GPIB)  {
        if (bd[instrID] > 0)
            if (CloseDev (bd[instrID]) < 0)  {
                tkvx433X_err = 221;
                return  -1;
            }
    }
       else 
        if (CloseVXIlibrary () < 0) {
            tkvx433X_err = 221;
            return  -1;
        }

    log_addr[instrID] = 0;
    bd[instrID] = 0;
    base_addr[instrID] = 0L;
    interface[instrID] = 0;
    instr_cnt--;
    return 0;
}

/*  =========================================================================  */
/*                           NI-GPIB ONLY                                      */
/*   Function: Initialize GPIB                                                 */
/*   Purpose:  This function initializes the GPIB interface.  First, the       */
/*             GPIB-VXI interface is initialized, then the Instrument          */
/*             interface is initialized.  The primary address of all entries   */
/*             is set to 1.  The secondary address the GPIB-VXI is 0 and the   */
/*             secondary address of the instrument is obtained by querying     */
/*             the GPIB-VXI.  The return value of the function is equal to     */
/*             the global error variable.                                      */
/*  =========================================================================  */
int tkvx433X_init_gpib (int instrID)
{
    int sec_addr;

    tkvx433X_err = 0;
    /*   If the GPIB-VXI is not opened (bd[0]=0), open it and initialize its        */
    /*   primary and secondary addresses.                                           */
    if (bd[0] <= 0)  {
        CloseInstrDevs ("tkvx433X");
        /*   Set the interface of the GPIB-VXI.  */
        interface[0] = NI_GPIB;
        bd[0] = OpenDev ("", "tkvx433X");
        if (bd[0] <= 0)  {
            tkvx433X_err = 220;
            return tkvx433X_err;
        }
        if (ibpad (bd[0], 1) & 0x8000)  {
            tkvx433X_err = 233;
            return tkvx433X_err;
        }
        if (ibsad (bd[0], 0x60) & 0x8000)  {
            tkvx433X_err = 233;
            return tkvx433X_err;
        }
    }
    /*   If the instrument is not opened (bd[instrID]=0), open it and initialize    */
    /*   its primary and secondary addresses.                                       */
    if (bd[instrID] <= 0)  {
        bd[instrID] = OpenDev ("", "tkvx433X");
        if (bd[instrID] <= 0)  {
            tkvx433X_err = 220;
            return tkvx433X_err;
        }
    }
    /*   Query the GPIB-VXI for the secondary address of the instrument.   */
    Fmt (cmd, "%s<LaSaddr? %d[b4]", log_addr[instrID]);
    if (tkvx433X_write_data (0, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    if (tkvx433X_read_data (0, cmd, 50) != 0)
        return tkvx433X_err;
    if (Scan (cmd, "%s>%d[b4]", &sec_addr) != 1)  {
        tkvx433X_err = 236;
        return tkvx433X_err;
    }
    if (ibpad (bd[instrID], 1) & 0x8000)  {
        tkvx433X_err = 233;
        return tkvx433X_err;
    }
    if (sec_addr != 255)  {
        if (ibsad (bd[instrID], sec_addr + 0x60) & 0x8000)  {
            tkvx433X_err = 233;
            return tkvx433X_err;
        }
    }
    else  if (ibsad (bd[instrID], 0x60) & 0x8000)  {
        tkvx433X_err = 233;
        return tkvx433X_err;
    }
    if (ibconfig (bd[instrID], 6, 1) & 0x8000)  {
        tkvx433X_err = 241;
        return tkvx433X_err;
    }
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Invalid Integer Range                                           */
/*   Purpose:  This function checks an integer to see if it lies between a     */
/*             minimum and maximum value.  If the value is out of range, set   */
/*             the global error variable to the value err_code.  If the        */
/*             value is OK, error = 0.  The return value is equal to the       */
/*             global error value.                                             */
/*  =========================================================================  */
int tkvx433X_invalid_integer_range (int val, int min, int max, int err_code)
{

    if (val < min || val > max)
        tkvx433X_err = err_code;
    else
        tkvx433X_err = 0;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Invalid Long Integer Range                                      */
/*   Purpose:  This function checks a long integer to see if it lies between   */
/*             a minimum and maximum value.  If the value is out of range,     */
/*             set the global error variable to the value err_code.  If the    */
/*             value is OK, error = 0.  The return value is equal to the       */
/*             global error value.                                             */
/*  =========================================================================  */
int tkvx433X_invalid_longint_range (long val, long min, long max, int err_code)
{

    if (val < min || val > max)
        tkvx433X_err = err_code;
    else
        tkvx433X_err = 0;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Device Closed                                                   */
/*   Purpose:  This function checks to see if the instrument has been          */
/*             initialized. The return value is equal to the global error      */
/*             variable.                                                       */
/*  =========================================================================  */
int tkvx433X_device_closed (int instrID)
{

    if (log_addr[instrID] <= 0)
        tkvx433X_err = 232;
    else
        tkvx433X_err = 0;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Read Data                                                       */
/*   Purpose:  This function reads a buffer of data from the instrument. The   */
/*             return value is equal to the global error variable.             */
/*  =========================================================================  */
int tkvx433X_read_data (int instrID, char *buf, int cnt)
{
    unsigned long bytes;

    tkvx433X_err = 0;
    switch (interface[instrID])  {
    case NI_VXI:
        if (WSrd (log_addr[instrID], (unsigned char *)buf, (long)cnt, 1, &bytes) & 0x8000)
            tkvx433X_err = 231;
        break;
    case NI_GPIB:
        if (ibrd (bd[instrID], buf, (long)cnt) & 0x8000)
            tkvx433X_err = 231;
        break;
    default:
        tkvx433X_err = 231;
        break;
    }
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Write Data                                                      */
/*   Purpose:  This function writes a buffer of data to the instrument. The    */
/*             return value is equal to the global error variable.             */
/*  =========================================================================  */
int tkvx433X_write_data (int instrID, char *buf, int cnt)
{
    unsigned long bytes;

    tkvx433X_err = 0;
    switch (interface[instrID])  {
    case NI_VXI:
        if (WSwrt (log_addr[instrID], (unsigned char *)buf, (long)cnt, 3, &bytes) & 0x8000 || bytes != (long)cnt)
            tkvx433X_err = 230;
        break;
    case NI_GPIB:
        if (ibwrt (bd[instrID], buf, (long)cnt) & 0x8000)
            tkvx433X_err = 230;
        break;
    default:
        tkvx433X_err = 230;
        break;
    }
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Read Register                                                   */
/*   Purpose:  This function reads an instrument register.  The return value   */
/*             is equal to the global error variable.                          */
/*  =========================================================================  */
int tkvx433X_read_reg (int instrID, int rel_addr, int *value)
{

    unsigned short temp_value = 0;

    tkvx433X_err = 0;
    switch (interface[instrID])  {
    case NI_VXI:
        if (VXIinReg (log_addr[instrID], rel_addr, &temp_value) < 0)
            tkvx433X_err = 231;
        *value = (int)temp_value;
        break;
    case NI_GPIB:
        Fmt (cmd, "RREG? %d[b4],%d[b4]", log_addr[instrID], rel_addr);
        if (ibwrt (bd[0], cmd, (long)NumFmtdBytes ()) & 0x8000)
            tkvx433X_err = 230;
        if (ibrd (bd[0], cmd, 20L) & 0x8000)
            tkvx433X_err = 231;
        if (Scan (cmd, "%s>%x[b4]", value) != 1)
            tkvx433X_err = 236;
        break;
    default:
        tkvx433X_err = 231;
        break;
    }
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Setup Arrays                                                    */
/*   Purpose:  This function is called by the init routine to initialize       */
/*             static arrays.  This routine should be modified for each        */
/*             instrument to include instrument dependent command arrays.      */
/*  =========================================================================  */
void tkvx433X_setup_arrays (void)
{

    /*   Timeout array values are equivalent to timeout in milliseconds.   */
    timeout[0] = 0xffffffffL;
    /*   Equivalent to turning timeout off   */
    timeout[1] = 1L;
    timeout[10] = 300L;
    timeout[2] = 1L;
    timeout[11] = 1000L;
    timeout[3] = 1L;
    timeout[12] = 3000L;
    timeout[4] = 1L;
    timeout[13] = 10000L;
    timeout[5] = 1L;
    timeout[14] = 30000L;
    timeout[6] = 3L;
    timeout[15] = 100000L;
    timeout[7] = 10L;
    timeout[16] = 300000L;
    timeout[8] = 30L;
    timeout[17] = 1000000L;
    timeout[9] = 100L;
    delaytime[0] = "set open_time";
    switching[0] = "open";
    delaytime[1] = "set open_time all";
    switching[1] = "openall;";
    delaytime[2] = "set close_time";
    switching[2] = "close";
    delaytime[3] = "set close_time all";
    interrupts[0] = "user_interrupt 1";
    interrupts[1] = "user_interrupt 2";
    interrupts[2] = "interrupt relay_closed";
    interrupts[3] = "interrupt program_done";
    interrupts[4] = "interrupt error";
    interrupts[5] = "interrupt trigger_in";
    trig_type[0] = "set trigger in";
    trig_type[1] = "set trigger in disable;";
    trig_type[2] = "set trigger out";
    trig_type[3] = "set trigger out disable;";
    trig_type[4] = "trigger";
    trig_type[5] = "trigger;";
    trig_type[6] = "trigger wait;";
}

/*  =========================================================================  */
/*   Function: Count Modules                                                   */
/*   Purpose:  This function counts the number of slave modules connected to   */
/*             the particular master scanner being initialized.                */
/*  =========================================================================  */
int tkvx433X_count_modules (int instrID, int *module_cnt)
{
    int count;

    Fmt (cmd, "%s<list? module all;");
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    if (tkvx433X_read_data (instrID, cmd, 100) != 0)
        return tkvx433X_err;
    count =  -1;
    while (cmd[0] != 32)  {
        count++;
        if (tkvx433X_read_data (instrID, cmd, 100) != 0)
            return tkvx433X_err;
    }
    *module_cnt = count;
    return tkvx433X_err;
}

/*  =========================================================================  */
/*   Function: Find Relay Range                                                */
/*   Purpose:  This function counts the total number of relays a module has    */
/*  =========================================================================  */
int tkvx433X_find_relay_range (int instrID, int module, int *total_relays)
{
    int relays_per_channel;

    /*    check if the relay is out of range for the specific module   */
    Fmt (cmd, "%s<list? module %d[b4];", module);
    if (tkvx433X_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tkvx433X_err;
    /*   Read back the information on this module.   */
    if (tkvx433X_read_data (instrID, cmd, 100) != 0)
        return tkvx433X_err;
    Scan (cmd, "%s>%s[t-dw31]%d[b1]", &relays_per_channel);
    *total_relays = 2 * relays_per_channel - 1;
    return tkvx433X_err;
}

